<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Services;
use App\Models\Transaction;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Notification;
use App\Notifications\PaymentReceived;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CheckoutController extends Controller
{
    public function index()
    {
        $cart = session('cart', []);
        
        if (empty($cart)) {
            return redirect()->route('services.index')->with('warning', 'Your cart is empty!');
        }

        $services = Services::whereIn('id', $cart)->get();
        $total = $services->sum('price');

        return view('checkout.index', compact('services', 'total'));
    }

    public function store(Request $request)
    {
        // Base validation rules
        $rules = [
            'user_name' => 'required|string|max:255',
            'user_phone' => 'required|string|max:20',
            'payment_method' => 'required|string|in:mobile_money,card,bank_transfer',
        ];

        // Dynamic validation based on payment method
        switch ($request->payment_method) {
            case 'mobile_money':
                $rules['mobile_money_phone'] = 'required|string|max:20';
                break;
            
            case 'card':
                $rules['card_name'] = 'required|string|max:255';
                $rules['card_number'] = 'required|string|min:13|max:19';
                $rules['card_expiry'] = 'required|string|size:5|regex:/^(0[1-9]|1[0-2])\/\d{2}$/';
                $rules['card_cvv'] = 'required|string|min:3|max:4';
                break;
            
            case 'bank_transfer':
                // No additional fields required for bank transfer
                break;
        }

        $request->validate($rules, [
            'mobile_money_phone.required' => 'Mobile money phone number is required.',
            'card_name.required' => 'Cardholder name is required.',
            'card_number.required' => 'Card number is required.',
            'card_number.min' => 'Card number must be at least 13 digits.',
            'card_expiry.required' => 'Card expiry date is required.',
            'card_expiry.regex' => 'Card expiry date must be in MM/YY format.',
            'card_cvv.required' => 'CVV is required.',
        ]);

        $cart = session('cart', []);
        
        if (empty($cart)) {
            return redirect()->route('services.index')->with('error', 'Your cart is empty!');
        }

        $services = Services::whereIn('id', $cart)->get();
        $totalAmount = $services->sum('price');

        // Generate unique transaction ID
        $transactionId = 'TXN-' . strtoupper(Str::random(10));

        try {
            DB::beginTransaction();

            // Prepare transaction data
            $transactionData = [
                'transaction_id' => $transactionId,
                'user_name' => $request->user_name,
                'user_phone' => $request->user_phone,
                'total_amount' => $totalAmount,
                'payment_method' => $request->payment_method,
                'status' => 'completed', // Simulate successful payment
            ];

            // Add payment method specific data
            switch ($request->payment_method) {
                case 'mobile_money':
                    $transactionData['mobile_money_phone'] = $request->mobile_money_phone;
                    $transactionData['payment_details'] = [
                        'mobile_money_phone' => $request->mobile_money_phone,
                        'provider' => 'Auto-detected', // Could be enhanced to detect provider
                    ];
                    break;
                
                case 'card':
                    $cardNumber = str_replace(' ', '', $request->card_number);
                    $transactionData['card_holder_name'] = $request->card_name;
                    $transactionData['card_last_four'] = substr($cardNumber, -4);
                    $transactionData['payment_details'] = [
                        'card_holder_name' => $request->card_name,
                        'card_last_four' => substr($cardNumber, -4),
                        'card_expiry' => $request->card_expiry,
                        'card_type' => $this->detectCardType($cardNumber),
                    ];
                    break;
                
                case 'bank_transfer':
                    $transactionData['payment_details'] = [
                        'transfer_type' => 'manual',
                        'bank_name' => 'SindaSystem Bank',
                        'account_number' => '1234567890',
                        'reference' => $request->user_name . '_' . $request->user_phone,
                    ];
                    break;
            }

            // Create transaction
            $transaction = Transaction::create($transactionData);

            // Attach services to transaction with their current prices
            foreach ($services as $service) {
                $transaction->services()->attach($service->id, [
                    'price' => $service->price,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }

            DB::commit();

            // Log successful transaction with payment method details
            $this->logTransactionSuccess($transaction, $request);

            // Send SMS notification
            try {
                Notification::route('log', null)
                    ->notify(new PaymentReceived($request->user_name, $totalAmount, $transactionId));
            } catch (\Exception $e) {
                Log::error('SMS notification failed: ' . $e->getMessage());
            }

            // Clear cart
            session()->forget('cart');

            return view('checkout.success', compact('transactionId', 'transaction'));

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Transaction failed: ' . $e->getMessage(), [
                'user_name' => $request->user_name,
                'payment_method' => $request->payment_method,
                'total_amount' => $totalAmount,
                'error' => $e->getMessage()
            ]);
            
            return redirect()->back()
                ->withInput()
                ->with('error', 'Transaction failed. Please try again.');
        }
    }

    private function detectCardType($cardNumber)
    {
        $cardNumber = preg_replace('/\D/', '', $cardNumber);
        
        if (preg_match('/^4/', $cardNumber)) {
            return 'Visa';
        } elseif (preg_match('/^5[1-5]/', $cardNumber)) {
            return 'Mastercard';
        } elseif (preg_match('/^3[47]/', $cardNumber)) {
            return 'American Express';
        } elseif (preg_match('/^6(?:011|5)/', $cardNumber)) {
            return 'Discover';
        }
        
        return 'Unknown';
    }

    private function logTransactionSuccess($transaction, $request)
    {
        $logData = [
            'transaction_id' => $transaction->transaction_id,
            'customer_name' => $transaction->user_name,
            'customer_phone' => $transaction->user_phone,
            'total_amount' => $transaction->total_amount,
            'payment_method' => $transaction->payment_method,
            'services_count' => $transaction->services()->count(),
            'timestamp' => now()->toDateTimeString(),
        ];

        // Add payment method specific details to log
        switch ($request->payment_method) {
            case 'mobile_money':
                $logData['mobile_money_phone'] = $request->mobile_money_phone;
                Log::info('🎉 MOBILE MONEY PAYMENT SUCCESSFUL! 📱', $logData);
                Log::info("💰 Payment of $" . number_format($transaction->total_amount, 2) . " received via Mobile Money from {$transaction->user_name} ({$request->mobile_money_phone})");
                Log::info("📱 Mobile Money Details: Phone: {$request->mobile_money_phone} | Transaction ID: {$transaction->transaction_id}");
                break;
            
            case 'card':
                $maskedCardNumber = '****-****-****-' . substr(str_replace(' ', '', $request->card_number), -4);
                $logData['card_holder'] = $request->card_name;
                $logData['masked_card_number'] = $maskedCardNumber;
                $logData['card_type'] = $transaction->payment_details['card_type'] ?? 'Unknown';
                Log::info('🎉 CARD PAYMENT SUCCESSFUL! 💳', $logData);
                Log::info("💰 Payment of $" . number_format($transaction->total_amount, 2) . " received via {$logData['card_type']} Card ({$maskedCardNumber}) from {$request->card_name}");
                Log::info("💳 Card Details: Holder: {$request->card_name} | Last 4: " . substr(str_replace(' ', '', $request->card_number), -4) . " | Expiry: {$request->card_expiry}");
                break;
            
            case 'bank_transfer':
                Log::info('🎉 BANK TRANSFER INITIATED! 🏦', $logData);
                Log::info("💰 Bank transfer of $" . number_format($transaction->total_amount, 2) . " initiated by {$transaction->user_name}. Awaiting verification.");
                Log::info("🏦 Bank Transfer Details: Reference: {$transaction->user_name}_{$transaction->user_phone} | Account: 1234567890");
                break;
        }

        // Log services purchased
        $serviceNames = $transaction->services->pluck('name')->toArray();
        Log::info('📦 Services purchased: ' . implode(', ', $serviceNames));
        
        // Log success message - FIXED: Changed | to . for concatenation
        Log::info("✅ Transaction {$transaction->transaction_id} completed successfully! Customer: {$transaction->user_name} | Amount: $" . number_format($transaction->total_amount, 2) . " | Payment Method: {$transaction->payment_method_display}");
        
        // Log separator for readability
        Log::info("═══════════════════════════════════════════════════════════════");
    }
}
