<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Services;
use NumberFormatter;

class ServicesList extends Component
{
    public $services;
    public $cart = [];
    public $searchTerm = '';
    public $selectedCategory = '';

    public function mount()
    {
        $this->services = Services::where('status', 'active')->get();
        $this->cart = session('cart', []);
    }

    public function addToCart($serviceId)
    {
        if (!in_array($serviceId, $this->cart)) {
            $this->cart[] = $serviceId;
            session(['cart' => $this->cart]);
            
            // Emit event to update cart count
            $this->dispatch('cartUpdated', count($this->cart));
            
            // Show success message
            session()->flash('message', 'Service added to cart successfully!');
        } else {
            session()->flash('warning', 'Service is already in your cart!');
        }
    }

    public function removeFromCart($serviceId)
    {
        $this->cart = array_values(array_filter($this->cart, function($id) use ($serviceId) {
            return $id != $serviceId;
        }));
        
        session(['cart' => $this->cart]);
        $this->dispatch('cartUpdated', count($this->cart));
        session()->flash('message', 'Service removed from cart!');
    }

    public function clearSearch()
    {
        $this->searchTerm = '';
        $this->selectedCategory = '';
    }

    public function getCategoriesProperty()
    {
        return $this->services->pluck('category')->unique()->filter()->values()->toArray();
    }

    public function getFilteredServicesProperty()
    {
        $query = $this->services;

        if ($this->searchTerm) {
            $query = $query->filter(function ($service) {
                return stripos($service->name, $this->searchTerm) !== false ||
                       stripos($service->description, $this->searchTerm) !== false;
            });
        }

        if ($this->selectedCategory) {
            $query = $query->filter(function ($service) {
                return $service->category === $this->selectedCategory;
            });
        }

        return $query;
    }

    public function render()
    {
        // Set the locale to Zambian Kwacha
        $formatter = new NumberFormatter('en_ZM', NumberFormatter::CURRENCY);

        // Format the service prices
        $formattedServices = $this->getFilteredServicesProperty()->map(function ($service) use ($formatter) {
            $service->formatted_price = $formatter->formatCurrency($service->price, 'ZMW');
            return $service;
        });

        return view('livewire.services-list', [
            'services' => $formattedServices,
        ]);
    }
}
