<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\Log;

class PaymentReceived extends Notification
{
    use Queueable;

    private $userName;
    private $totalAmount;
    private $transactionId;

    /**
     * Create a new notification instance.
     */
    public function __construct($userName, $totalAmount, $transactionId)
    {
        $this->userName = $userName;
        $this->totalAmount = $totalAmount;
        $this->transactionId = $transactionId;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['log', 'database']; // You can add 'nexmo' or 'vonage' for real SMS
    }

    /**
     * Get the log representation of the notification.
     */
    public function toLog(object $notifiable): array
    {
        $message = "SMS SENT - Thank you {$this->userName}! Payment of \${$this->totalAmount} received. Transaction ID: {$this->transactionId}";
        
        Log::info($message);
        
        return [
            'message' => $message,
            'user_name' => $this->userName,
            'amount' => $this->totalAmount,
            'transaction_id' => $this->transactionId
        ];
    }

    /**
     * Get the database representation of the notification.
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'type' => 'payment_received',
            'message' => "Thank you {$this->userName}! Payment of \${$this->totalAmount} received.",
            'transaction_id' => $this->transactionId,
            'user_name' => $this->userName,
            'amount' => $this->totalAmount
        ];
    }

    /**
     * Get the SMS representation of the notification (for future SMS integration).
     */
    public function toNexmo(object $notifiable): array
    {
        return [
            'content' => "Thank you {$this->userName}! Payment of \${$this->totalAmount} received. Transaction ID: {$this->transactionId}. - SindaSystem"
        ];
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'user_name' => $this->userName,
            'total_amount' => $this->totalAmount,
            'transaction_id' => $this->transactionId,
            'message' => "Thank you {$this->userName}! Payment of \${$this->totalAmount} received. Transaction ID: {$this->transactionId}"
        ];
    }
}
